"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkHtmlElement = checkHtmlElement;
exports.parseCSS = parseCSS;
exports.deprecate = deprecate;
exports.getMessage = getMessage;
exports.matches = matches;
exports.normalize = normalize;
exports.getTag = getTag;
exports.getSingleElementValue = getSingleElementValue;
exports.compareArraysAsSet = compareArraysAsSet;
exports.HtmlElementTypeError = void 0;

var _redent = _interopRequireDefault(require("redent"));

var _jestMatcherUtils = require("jest-matcher-utils");

var _css = require("css");

var _isEqual = _interopRequireDefault(require("lodash/isEqual"));

class HtmlElementTypeError extends Error {
  constructor(received, matcherFn, context) {
    super();
    /* istanbul ignore next */

    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, matcherFn);
    }

    let withType = '';

    try {
      withType = (0, _jestMatcherUtils.printWithType)('Received', received, _jestMatcherUtils.printReceived);
    } catch (e) {// Can throw for Document:
      // https://github.com/jsdom/jsdom/issues/2304
    }

    this.message = [(0, _jestMatcherUtils.matcherHint)(`${context.isNot ? '.not' : ''}.${matcherFn.name}`, 'received', ''), '', `${(0, _jestMatcherUtils.RECEIVED_COLOR)('received')} value must be an HTMLElement or an SVGElement.`, withType].join('\n');
  }

}

exports.HtmlElementTypeError = HtmlElementTypeError;

function checkHasWindow(htmlElement, ...args) {
  if (!htmlElement || !htmlElement.ownerDocument || !htmlElement.ownerDocument.defaultView) {
    throw new HtmlElementTypeError(htmlElement, ...args);
  }
}

function checkHtmlElement(htmlElement, ...args) {
  checkHasWindow(htmlElement, ...args);
  const window = htmlElement.ownerDocument.defaultView;

  if (!(htmlElement instanceof window.HTMLElement) && !(htmlElement instanceof window.SVGElement)) {
    throw new HtmlElementTypeError(htmlElement, ...args);
  }
}

class InvalidCSSError extends Error {
  constructor(received, matcherFn) {
    super();
    /* istanbul ignore next */

    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, matcherFn);
    }

    this.message = [received.message, '', (0, _jestMatcherUtils.RECEIVED_COLOR)(`Failing css:`), (0, _jestMatcherUtils.RECEIVED_COLOR)(`${received.css}`)].join('\n');
  }

}

function parseCSS(css, ...args) {
  const ast = (0, _css.parse)(`selector { ${css} }`, {
    silent: true
  }).stylesheet;

  if (ast.parsingErrors && ast.parsingErrors.length > 0) {
    const {
      reason,
      line
    } = ast.parsingErrors[0];
    throw new InvalidCSSError({
      css,
      message: `Syntax error parsing expected css: ${reason} on line: ${line}`
    }, ...args);
  }

  const parsedRules = ast.rules[0].declarations.filter(d => d.type === 'declaration').reduce((obj, {
    property,
    value
  }) => Object.assign(obj, {
    [property]: value
  }), {});
  return parsedRules;
}

function display(value) {
  return typeof value === 'string' ? value : (0, _jestMatcherUtils.stringify)(value);
}

function getMessage(matcher, expectedLabel, expectedValue, receivedLabel, receivedValue) {
  return [`${matcher}\n`, `${expectedLabel}:\n${(0, _jestMatcherUtils.EXPECTED_COLOR)((0, _redent.default)(display(expectedValue), 2))}`, `${receivedLabel}:\n${(0, _jestMatcherUtils.RECEIVED_COLOR)((0, _redent.default)(display(receivedValue), 2))}`].join('\n');
}

function matches(textToMatch, matcher) {
  if (matcher instanceof RegExp) {
    return matcher.test(textToMatch);
  } else {
    return textToMatch.includes(String(matcher));
  }
}

function deprecate(name, replacementText) {
  // Notify user that they are using deprecated functionality.
  // eslint-disable-next-line no-console
  console.warn(`Warning: ${name} has been deprecated and will be removed in future updates.`, replacementText);
}

function normalize(text) {
  return text.replace(/\s+/g, ' ').trim();
}

function getTag(element) {
  return element.tagName && element.tagName.toLowerCase();
}

function getSelectValue({
  multiple,
  options
}) {
  const selectedOptions = [...options].filter(option => option.selected);

  if (multiple) {
    return [...selectedOptions].map(opt => opt.value);
  }
  /* istanbul ignore if */


  if (selectedOptions.length === 0) {
    return undefined; // Couldn't make this happen, but just in case
  }

  return selectedOptions[0].value;
}

function getInputValue(inputElement) {
  switch (inputElement.type) {
    case 'number':
      return inputElement.value === '' ? null : Number(inputElement.value);

    case 'checkbox':
      return inputElement.checked;

    default:
      return inputElement.value;
  }
}

function getSingleElementValue(element) {
  /* istanbul ignore if */
  if (!element) {
    return undefined;
  }

  switch (element.tagName.toLowerCase()) {
    case 'input':
      return getInputValue(element);

    case 'select':
      return getSelectValue(element);

    default:
      return element.value;
  }
}

function compareArraysAsSet(a, b) {
  if (Array.isArray(a) && Array.isArray(b)) {
    return (0, _isEqual.default)(new Set(a), new Set(b));
  }

  return undefined;
}